from pyparsing import Literal

from codeable_detectors.pyparsing_patterns import round_braces_block
from codeable_detectors.utils import get_all_files, read_file
from codeable_detectors.evidences import Evidences, FailedEvidence, Evidence
from codeable_detectors.detector_context import DetectorContext


class Detector(object):
    pass


class AtLeastOneFileMatchesDetector(Detector):
    def __init__(self):
        self.file_endings = []
        self.file_names = []

    def get_all_matching_files(self, directory):
        matching_files = []
        for file in get_all_files(directory):
            for name in self.file_names:
                if file.endswith(name):
                    matching_files.append(file)
            else:
                for ending in self.file_endings:
                    if file.endswith(ending):
                        matching_files.append(file)
        return matching_files

    def detect(self, directory, **kwargs):
        evidences = Evidences()
        # we use failed_evidences to gather all that have failed in the various files;
        # they are only used if no file has positive evidences, else those are returned
        # using evidences
        failed_evidences = Evidences()
        print("### DETECT in DIR: " + directory)
        for file_name in self.get_all_matching_files(directory):
            print("*** DETECT in FILE: " + file_name)
            ctx = DetectorContext(read_file(file_name), file_name)
            file_evidence = self.detect_in_context(ctx, **kwargs)
            if isinstance(file_evidence, FailedEvidence):
                file_evidence.prepend("in file " + file_name + ": ")
                failed_evidences.add(file_evidence)
            else:
                evidences.add(file_evidence)
        if evidences.is_empty():
            if failed_evidences.is_empty():
                failed_evidences.add(FailedEvidence("no matching file"))
            return failed_evidences
        return evidences

    def detect_in_context(self, ctx, **kwargs):
        raise NotImplementedError()


# apply multiple codeable_detectors across the files
class MatchesAcrossMultipleFilesDetector(Detector):
    def __init__(self, detectors):
        super().__init__()
        self.detectors = detectors

    def detect(self, directory, **kwargs):
        print("### MULTI-DETECT in DIR: " + directory)
        evidences = Evidences()
        for detector in self.detectors:
            failed_evidence_error_prepend = "detector '" + detector.__class__.__name__ + ": "
            detector_evidences = detector.detect(directory, **kwargs)
            if detector_evidences.is_empty():
                detector_evidences.add(FailedEvidence(failed_evidence_error_prepend + "no matching files"))
            else:
                evidences.add(detector_evidences)
        return evidences


class Invocation(AtLeastOneFileMatchesDetector):
    def __init__(self, operation_name, file_endings, object_name_pattern=None,
                 inside_invocation_patterns=None, closing_semicolon=True):
        super().__init__()
        if inside_invocation_patterns is None:
            inside_invocation_patterns = []
        if not isinstance(file_endings, list):
            file_endings = [file_endings]
        self.operationName = operation_name
        self.file_endings = file_endings
        if not isinstance(inside_invocation_patterns, list):
            inside_invocation_patterns = [inside_invocation_patterns]
        self.insidePatterns = inside_invocation_patterns
        self.invocationPattern = Literal(operation_name) + round_braces_block
        self.objectNamePattern = object_name_pattern
        if object_name_pattern:
            self.invocationPattern = object_name_pattern + Literal(".") + self.invocationPattern
        if closing_semicolon:
            self.invocationPattern = self.invocationPattern + Literal(";")

    def detect_in_context(self, ctx, **kwargs):
        matches = []
        line_matches = ctx.matches_pattern(self.invocationPattern)
        for match in line_matches:
            if self.insidePatterns:
                header_string_matches = DetectorContext(match).matches_patterns(self.insidePatterns)
                if header_string_matches:
                    matches.append(match)
            else:
                matches.append(match)
        if not matches:
            inside_string = ""
            if self.insidePatterns:
                inside_string = "with inside invocation patterns: '" + ", ".join(
                    str(s) for s in self.insidePatterns) + "' "
            if self.objectNamePattern:
                object_string = "object pattern: '" + self.objectNamePattern + "' method name: "
            else:
                object_string = " method name: '"
            return FailedEvidence("invocation to " + object_string + self.operationName + "' " + inside_string +
                                  "not found")
        print("### matches 1= " + str(matches))
        return Evidence(matches)


class FunctionInvocation(Invocation):
    def __init__(self, function_name, file_endings, inside_invocation_patterns=None, closing_semicolon=True):
        super().__init__(function_name, file_endings, inside_invocation_patterns=inside_invocation_patterns,
                         closing_semicolon=closing_semicolon)


class ObjectMethodInvocation(Invocation):
    def __init__(self, object_name_pattern, method_name, file_endings, inside_invocation_patterns=None, closing_semicolon=True):
        super().__init__(method_name, file_endings, inside_invocation_patterns=inside_invocation_patterns,
                         closing_semicolon=closing_semicolon, object_name_pattern=object_name_pattern)


def contains_url_to_alias(url_string, aliases):
    if "http://" in url_string:
        url_string = url_string[url_string.find("http://") + 7:]
    elif "https://" in url_string:
        url_string = url_string[url_string.find("https://") + 8:]
    else:
        return False
    url_string = url_string.lower()
    for alias in aliases:
        if alias.lower() in url_string:
            return True
    return False


def contains_alias(string, aliases):
    string = string.lower()
    for alias in aliases:
        if alias.lower() in string:
            return True
    return False


def get_var_assignment_matches_containing_url_alias(var_assignment_matches, aliases):
    vars_and_matches = []
    for var_assignment_match in var_assignment_matches:
        if contains_alias(var_assignment_match.text, aliases):
            var = var_assignment_match.text[:var_assignment_match.text.find("=")].strip()
            vars_and_matches.append([var, var_assignment_match])
    print("vars = " + str(vars_and_matches))
    return vars_and_matches
